/**
 * @author Toru Nagashima
 * @copyright 2016 Toru Nagashima. All rights reserved.
 * See LICENSE file in root directory for full license.
 */
"use strict";

//------------------------------------------------------------------------------
// Requirements
//------------------------------------------------------------------------------

var Promise = require("pinkie-promise");
var runAll = require("../../lib");
var parseCLIArgs = require("../common/parse-cli-args");

//------------------------------------------------------------------------------
// Public Interface
//------------------------------------------------------------------------------

/**
 * Parses arguments, then run specified npm-scripts.
 *
 * @param {string[]} args - Arguments to parse.
 * @param {stream.Writable} stdout - A writable stream to print logs.
 * @param {stream.Writable} stderr - A writable stream to print errors.
 * @returns {Promise} A promise which comes to be fulfilled when all npm-scripts are completed.
 * @private
 */
module.exports = function npmRunAll(args, stdout, stderr) {
    try {
        var stdin = process.stdin;

        var _parseCLIArgs = parseCLIArgs(args, { parallel: true }, { singleMode: true }),
            _parseCLIArgs$lastGro = _parseCLIArgs.lastGroup,
            patterns = _parseCLIArgs$lastGro.patterns,
            parallel = _parseCLIArgs$lastGro.parallel,
            continueOnError = _parseCLIArgs.continueOnError,
            config = _parseCLIArgs.config,
            packageConfig = _parseCLIArgs.packageConfig,
            printLabel = _parseCLIArgs.printLabel,
            printName = _parseCLIArgs.printName,
            silent = _parseCLIArgs.silent,
            race = _parseCLIArgs.race,
            yarn = _parseCLIArgs.yarn,
            rest = _parseCLIArgs.rest;

        if (patterns.length === 0) {
            return Promise.resolve(null);
        }

        return runAll(patterns, {
            stdout: stdout,
            stderr: stderr,
            stdin: stdin,
            parallel: parallel,
            continueOnError: continueOnError,
            printLabel: printLabel,
            printName: printName,
            config: config,
            packageConfig: packageConfig,
            silent: silent,
            arguments: rest,
            race: race,
            yarn: yarn
        });
    } catch (err) {
        return Promise.reject(err);
    }
};