/// <reference types="node"/>
/// <reference types="webdriver"/>

declare namespace WebdriverIO {
    function remote(
        options?: WebDriver.Options,
        modifier?: (...args: any[]) => any
    ): WebDriver.Client & WebdriverIO.Browser;

    function multiremote(
        options: any
    ): WebDriver.Client;

    type LocationParam = 'x' | 'y';

    interface LocationReturn {
        x: number,
        y: number
    }

    type SizeParam = 'width' | 'height';

    interface SizeReturn {
        width: number,
        height: number
    }

    interface Cookie {
        name: string,
        value: string,
        domain?: string,
        path?: string,
        expiry?: number,
        isSecure?: boolean,
        isHttpOnly?: boolean
    }

    interface CSSProperty {
        property: string,
        value: any,
        parsed?: {
            // other
            unit?: string,
            // font-family
            value?: any,
            string: string,
            // color
            hex?: string,
            alpha?: number,
            type?: string,
            rgba?: string
        }
    }

    interface Options {
        runner?: string,
        specs?: string[],
        exclude?: string[],
        suites?: object,
        maxInstances?: number,
        maxInstancesPerCapability?: number,
        capabilities?: WebDriver.DesiredCapabilities | WebDriver.DesiredCapabilities[],
        outputDir?: string,
        baseUrl?: string,
        bail?: number,
        waitforTimeout?: number,
        waitforInterval?: number,
        framework?: string,
        mochaOpts?: object,
        jasmineNodeOpts?: object,
        reporters?: (string | object)[],
        services?: (string | object)[],
        execArgv?: string[]
    }

    interface Suite {}
    interface Test {}

    interface Results {
        finished: number,
        passed: number,
        failed: number
    }

    interface Hooks {

        onPrepare?(
            config: Config,
            capabilities: WebDriver.DesiredCapabilities
        ): void;

        onComplete?(exitCode: number, config: Config, capabilities: WebDriver.DesiredCapabilities, results: Results): void;

        onReload?(oldSessionId: string, newSessionId: string): void;

        before?(
            capabilities: WebDriver.DesiredCapabilities,
            specs: string[]
        ): void;

        beforeCommand?(
            commandName: string,
            args: any[]
        ): void;

        beforeHook?(): void;

        beforeSession?(
            config: Config,
            capabilities: WebDriver.DesiredCapabilities,
            specs: string[]
        ): void;

        beforeSuite?(suite: Suite): void;
        beforeTest?(test: Test): void;
        afterHook?(): void;

        after?(
            result: number,
            capabilities: WebDriver.DesiredCapabilities,
            specs: string[]
        ): void;

        afterCommand?(
            commandName: string,
            args: any[],
            result: any,
            error?: Error
        ): void;

        afterSession?(
            config: Config,
            capabilities: WebDriver.DesiredCapabilities,
            specs: string[]
        ): void;

        afterSuite?(suite: Suite): void;
        afterTest?(test: Test): void;

        // cucumber specific hooks
        beforeFeature?(feature: string): void;
        beforeScenario?(scenario: string): void;
        beforeStep?(step: string): void;
        afterFeature?(feature: string): void;
        afterScenario?(scenario: any): void;
        afterStep?(stepResult: any): void;
    }

    type ActionTypes = 'press' | 'longPress' | 'tap' | 'moveTo' | 'wait' | 'release';
    interface TouchAction {
        action: ActionTypes,
        x?: number,
        y?: number,
        element?: Element
    }
    type TouchActions = string | TouchAction | TouchAction[];

    interface Element {
        addCommand(
            name: string,
            func: Function
        ): undefined;
        $$(
            selector: string | Function
        ): Element[];
        $(
            selector: string | Function
        ): Element;
        addValue(
            value: any
        ): undefined;
        clearValue(): undefined;
        click(): undefined;
        doubleClick(): undefined;
        dragAndDrop(
            target: Element,
            duration?: number
        ): undefined;
        getAttribute(
            attributeName: string
        ): string;
        getCSSProperty(
            cssProperty: string
        ): CSSProperty;
        getHTML(
            includeSelectorTag?: boolean
        ): string;
        getLocation(
            prop: LocationParam
        ): number;
        getLocation(): LocationReturn;
        getProperty(
            property: string
        ): object | string;
        getSize(
            prop: SizeParam
        ): number;
        getSize(): SizeReturn;
        getTagName(): string;
        getText(): string;
        getValue(): string;
        isDisplayed(): boolean;
        isDisplayedInViewport(): boolean;
        isEnabled(): boolean;
        isExisting(): boolean;
        isFocused(): boolean;
        isSelected(): boolean;
        moveTo(
            xoffset?: number,
            yoffset?: number
        ): undefined;
        saveScreenshot(
            filename: string
        ): Buffer;
        scrollIntoView(): undefined;
        selectByAttribute(
            attribute: string,
            value: string
        ): undefined;
        selectByIndex(
            index: number
        ): undefined;
        selectByVisibleText(
            text: string
        ): undefined;
        setValue(
            value: any
        ): undefined;
        shadow$$(): undefined;
        shadow$(): undefined;
        touchAction(
            action: TouchActions
        ): undefined;
        waitForDisplayed(
            ms?: number,
            reverse?: boolean,
            error?: string
        ): undefined;
        waitForEnabled(
            ms?: number,
            reverse?: boolean,
            error?: string
        ): undefined;
        waitForExist(
            ms?: number,
            reverse?: boolean,
            error?: string
        ): undefined;
    }

    type Execute = <T>(script: string | ((...arguments: any[]) => T), ...arguments: any[]) => T;
    type ExecuteAsync = (script: string | ((...arguments: any[]) => any), ...arguments: any[]) => any;
    type Call = <T>(callback: Function) => T;
    interface Timeouts {
        implicit?: number,
        pageLoad?: number,
        script?: number
    }

    interface Browser {
        addCommand(
            name: string,
            func: Function,
            attachToElement?: boolean
        ): undefined;
        execute: Execute;
        executeAsync: ExecuteAsync;
        call: Call;
        options: Options;
        waitUntil(
            condition: () => boolean,
            timeout?: number,
            timeoutMsg?: string,
            interval?: number
        ): undefined
        $$(
            selector: string | Function
        ): Element[];
        $(
            selector: string | Function
        ): Element;
        debug(): undefined;
        deleteCookies(
            names?: string[]
        ): undefined;
        getCookies(
            names?: string[]
        ): Cookie[];
        keys(
            value: string | string[]
        ): undefined;
        newWindow(
            url: string,
            windowName?: string,
            windowFeatures?: string
        ): string;
        pause(
            milliseconds: number
        ): undefined;
        reloadSession(): undefined;
        saveScreenshot(
            filepath: string
        ): Buffer;
        setCookies(
            cookie: Cookie
        ): undefined;
        setTimeout(
            timeouts: Timeouts
        ): undefined;
        switchWindow(
            urlOrTitleToMatch: string | RegExp
        ): undefined;
        touchAction(
            action: TouchActions
        ): undefined;
        url(
            url?: string
        ): string;
    }

    type Omit<T, K extends keyof T> = Pick<T, Exclude<keyof T, K>>;

    interface Config extends Options, Omit<WebDriver.Options, "capabilities">, Hooks {}
}

declare var browser: WebDriver.Client & WebdriverIO.Browser;
declare function $(selector: string | Function): WebdriverIO.Element;
declare function $$(selector: string | Function): WebdriverIO.Element[];

declare module "webdriverio" {
    export = WebdriverIO
}
