"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.multiremoteHandler = exports.elementErrorHandler = void 0;

var _logger = _interopRequireDefault(require("@wdio/logger"));

var _refetchElement = _interopRequireDefault(require("./utils/refetchElement"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const log = (0, _logger.default)('webdriverio');
/**
 * This method is an command wrapper for elements that checks if a command is called
 * that wasn't found on the page and automatically waits for it
 *
 * @param  {Function} fn  commandWrap from wdio-sync package (or shim if not running in sync)
 */

const elementErrorHandler = fn => (commandName, commandFn) => {
  return function (...args) {
    return fn(commandName, async () => {
      /**
       * wait on element if:
       *  - elementId couldn't be fetched in the first place
       *  - command is not explicit wait command for existance or displayedness
       */
      if (!this.elementId && !commandName.match(/(wait(Until|ForDisplayed|ForExist|ForEnabled)|isExisting|isDisplayed)/)) {
        log.debug(`command ${commandName} was called on an element ("${this.selector}") ` + 'that wasn\'t found, waiting for it...');
        /**
         * create new promise so we can apply a custom error message in cases waitForExist fails
         */

        try {
          await this.waitForExist();
          /**
           * if waitForExist was successful requery element and assign elementId to the scope
           */

          const element = await this.parent.$(this.selector);
          this.elementId = element.elementId;
        } catch (_unused) {
          throw new Error(`Can't call ${commandName} on element with selector "${this.selector}" because element wasn't found`);
        }
      }

      try {
        return await fn(commandName, commandFn).apply(this, args);
      } catch (error) {
        if (error.name === 'stale element reference') {
          const element = await (0, _refetchElement.default)(this);
          this.elementId = element.elementId;
          this.parent = element.parent;
          return await fn(commandName, commandFn).apply(this, args);
        }

        throw error;
      }
    }).apply(this);
  };
};
/**
 * handle single command calls from multiremote instances
 */


exports.elementErrorHandler = elementErrorHandler;

const multiremoteHandler = wrapCommand => commandName => {
  return wrapCommand(commandName, function (...args) {
    const commandResults = this.instances.map(instanceName => {
      return this[instanceName][commandName](...args);
    });
    return Promise.all(commandResults);
  });
};

exports.multiremoteHandler = multiremoteHandler;