"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = $$;

var _webdriver = require("webdriver");

var _config = require("@wdio/config");

var _lodash = _interopRequireDefault(require("lodash.merge"));

var _utils = require("../../utils");

var _middlewares = require("../../middlewares");

var _constants = require("../../constants");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * The `$$` command is a short way to call the [`findElements`](/docs/api/webdriver.html#findelements) command in order
 * to fetch multiple elements on the page similar to the `$$` command from the browser scope. The difference when calling
 * it from an element scope is that the driver will look within the children of that element.
 *
 * For more information on how to select specific elements, see [`Selectors`](/docs/selectors.html).
 *
 * <example>
    :index.html
    <ul id="menu">
        <li><a href="/">Home</a></li>
        <li><a href="/">Developer Guide</a></li>
        <li><a href="/">API</a></li>
        <li><a href="/">Contribute</a></li>
    </ul>
    :$.js
    it('should get text a menu link', () => {
        const text = $('#menu');
        console.log(text.$$('li')[2].$('a').getText()); // outputs: "API"
    });

    it('should get text a menu link - JS Function', () => {
        const text = $('#menu');
        console.log(text.$$(function() { // Arrow function is not allowed here.
            // this is Element https://developer.mozilla.org/en-US/docs/Web/API/Element
            // in this particular example it is HTMLUListElement
            // TypeScript users may do something like this
            // return (this as Element).querySelectorAll('li')
            return this.querySelectorAll('li'); // Element[]
        })[2].$('a').getText()); // outputs: "API"
    });
 * </example>
 *
 * @alias $$
 * @param {String|Function} selector  selector or JS Function to fetch multiple elements
 * @return {Element[]}
 * @type utility
 *
 */
async function $$(selector) {
  const res = await _utils.findElements.call(this, selector);
  const browser = (0, _utils.getBrowserObject)(this);
  const prototype = (0, _lodash.default)({}, browser.__propertiesObject__, (0, _utils.getPrototype)('element'), {
    scope: 'element'
  });
  const elements = res.map((res, i) => {
    const element = (0, _webdriver.webdriverMonad)(this.options, client => {
      const elementId = (0, _utils.getElementFromResponse)(res);

      if (elementId) {
        /**
         * set elementId for easy access
         */
        client.elementId = elementId;
        /**
         * set element id with proper key so element can be passed into execute commands
         */

        if (this.isW3C) {
          client[_constants.ELEMENT_KEY] = elementId;
        } else {
          client.ELEMENT = elementId;
        }
      } else {
        client.error = res;
      }

      client.selector = selector;
      client.parent = this;
      client.index = i;
      client.emit = this.emit.bind(this);
      return client;
    }, prototype);
    const elementInstance = element(this.sessionId, (0, _middlewares.elementErrorHandler)(_config.wrapCommand));
    const origAddCommand = elementInstance.addCommand.bind(elementInstance);

    elementInstance.addCommand = (name, fn) => {
      browser.__propertiesObject__[name] = {
        value: fn
      };
      origAddCommand(name, (0, _config.runFnInFiberContext)(fn));
    };

    return elementInstance;
  });
  return elements;
}