"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = $;

var _webdriver = require("webdriver");

var _config = require("@wdio/config");

var _lodash = _interopRequireDefault(require("lodash.merge"));

var _utils = require("../../utils");

var _middlewares = require("../../middlewares");

var _constants = require("../../constants");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * The `$` command is a short way to call the [`findElement`](/docs/api/webdriver.html#findelement) command in order
 * to fetch a single element on the page. It returns an object that with an extended prototype to call
 * action commands without passing in a selector. However if you still pass in a selector it will look
 * for that element first and call the action on that element.
 *
 * Using the wdio testrunner this command is a global variable else it will be located on the browser object instead.
 *
 * You can chain `$` or `$$` together in order to walk down the DOM tree. For more information on how
 * to select specific elements, see [`Selectors`](/docs/selectors.html).
 *
 * <example>
    :index.html
    <ul id="menu">
        <li><a href="/">Home</a></li>
        <li><a href="/">Developer Guide</a></li>
        <li><a href="/">API</a></li>
        <li><a href="/">Contribute</a></li>
    </ul>
    :$.js
    it('should get text a menu link', () => {
        const text = $('#menu');
        console.log(text.$$('li')[2].$('a').getText()); // outputs: "API"
    });

    it('should get text a menu link - JS Function', () => {
        const text = $(function() { // Arrow function is not allowed here.
            // this is Window https://developer.mozilla.org/en-US/docs/Web/API/Window
            // TypeScript users may do something like this
            // return (this as Window).document.querySelector('#menu')
            return this.document.querySelector('#menu'); // Element
        });
        console.log(text.$$('li')[2].$('a').getText()); // outputs: "API"
    });
 * </example>
 *
 * @alias $
 * @param {String|Function} selector  selector or JS Function to fetch a certain element
 * @return {Element}
 * @type utility
 *
 */
async function $(selector) {
  const res = await _utils.findElement.call(this, selector);
  const prototype = (0, _lodash.default)({}, this.__propertiesObject__, (0, _utils.getPrototype)('element'), {
    scope: 'element'
  });
  const element = (0, _webdriver.webdriverMonad)(this.options, client => {
    const elementId = (0, _utils.getElementFromResponse)(res);

    if (elementId) {
      /**
       * set elementId for easy access
       */
      client.elementId = elementId;
      /**
       * set element id with proper key so element can be passed into execute commands
       */

      if (this.isW3C) {
        client[_constants.ELEMENT_KEY] = elementId;
      } else {
        client.ELEMENT = elementId;
      }
    } else {
      client.error = res;
    }

    client.selector = selector;
    client.parent = this;
    client.emit = this.emit.bind(this);
    return client;
  }, prototype);
  const elementInstance = element(this.sessionId, (0, _middlewares.elementErrorHandler)(_config.wrapCommand));
  const origAddCommand = elementInstance.addCommand.bind(elementInstance);

  elementInstance.addCommand = (name, fn) => {
    this.__propertiesObject__[name] = {
      value: fn
    };
    origAddCommand(name, (0, _config.runFnInFiberContext)(fn));
  };

  return elementInstance;
}