"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Lint = require("tslint");
const ts = require("typescript");
const Helpers_1 = require("./Helpers");
class Options {
}
Options.VARIABLES = 'variables';
Options.PROPERTIES = 'properties';
exports.Options = Options;
class Rule extends Lint.Rules.AbstractRule {
    apply(sourceFile) {
        let result = new Array();
        return this.applyWithWalker(new NoUninitializedVariableWalker(sourceFile, this.getOptions()))
            .concat(this.applyWithWalker(new NoUninitializedPropertiesWalker(sourceFile, this.getOptions())));
    }
}
exports.Rule = Rule;
class NoUninitializedVariableWalker extends Lint.RuleWalker {
    visitModuleDeclaration(node) {
        return;
    }
    visitVariableDeclaration(node) {
        super.visitVariableDeclaration(node);
        if (super.hasOption(Options.VARIABLES)) {
            if (node.initializer === undefined
                && !Helpers_1.isUndefinedInDomainOf(node.type)
                && !Helpers_1.isDeclaredInForStatement(node)
                && !Helpers_1.isDeclaredInCatch(node)) {
                super.addFailureAt(node.getStart(), node.getEnd(), `Variable '${node.name.getText()}' is uninitialized. 'undefined' is not assignable to its type.`);
            }
        }
    }
}
class NoUninitializedPropertiesWalker extends Lint.RuleWalker {
    constructor() {
        super(...arguments);
        this._initializedProperties = [];
    }
    visitModuleDeclaration(node) {
        return;
    }
    visitClassLikeDeclaration(node, superVisitClassLikeDeclaration) {
        if (!super.hasOption(Options.PROPERTIES)) {
            return;
        }
        const _currentClassInitializedProperties = [];
        const constructor = Helpers_1.findConstructor(node.members);
        if (constructor && constructor.body) {
            for (const statement of constructor.body.statements) {
                if (statement.kind !== ts.SyntaxKind.ExpressionStatement) {
                    continue;
                }
                const expressionStatement = statement;
                if (expressionStatement.expression.kind !== ts.SyntaxKind.BinaryExpression) {
                    continue;
                }
                const binaryExpression = expressionStatement.expression;
                if (binaryExpression.left.kind !== ts.SyntaxKind.PropertyAccessExpression) {
                    continue;
                }
                const leftExpression = binaryExpression.left;
                if (leftExpression.expression.kind === ts.SyntaxKind.ThisKeyword) {
                    _currentClassInitializedProperties.push(leftExpression.name.getText());
                }
            }
        }
        this._initializedProperties.push(_currentClassInitializedProperties);
        superVisitClassLikeDeclaration(node);
        this._initializedProperties.pop();
    }
    visitClassDeclaration(node) {
        this.visitClassLikeDeclaration(node, (x) => super.visitClassDeclaration(x));
    }
    visitClassExpression(node) {
        this.visitClassLikeDeclaration(node, (x) => super.visitClassExpression(x));
    }
    visitPropertyDeclaration(node) {
        super.visitPropertyDeclaration(node);
        if (!super.hasOption(Options.PROPERTIES)) {
            return;
        }
        if (this.isNotInitialized(node) && Helpers_1.canNotBeUndefined(node)) {
            this.addFailureAt(node.getStart(), node.getEnd(), `Property '${node.name.getText()}' is never initialized. 'undefined' is not assignable to its type.`);
        }
    }
    isNotInitialized(node) {
        const isNotAssigned = !(this._initializedProperties[this._initializedProperties.length - 1].indexOf(node.name.getText()) !== -1);
        return !node.initializer && isNotAssigned;
    }
}
//# sourceMappingURL=noUninitializedRule.js.map