"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findAndReadConfig = findAndReadConfig;
exports.orNullIfFileNotExist = orNullIfFileNotExist;
exports.orIfFileNotExist = orIfFileNotExist;
exports.loadConfig = loadConfig;
exports.getConfig = getConfig;
exports.loadParentConfig = loadParentConfig;
exports.validateConfig = validateConfig;
exports.loadEnv = loadEnv;

function _fsExtra() {
  const data = require("fs-extra");

  _fsExtra = function () {
    return data;
  };

  return data;
}

function _jsYaml() {
  const data = require("js-yaml");

  _jsYaml = function () {
    return data;
  };

  return data;
}

var path = _interopRequireWildcard(require("path"));

function _ajv() {
  const data = _interopRequireDefault(require("ajv"));

  _ajv = function () {
    return data;
  };

  return data;
}

function _ajvErrorNormalizer() {
  const data = require("./ajvErrorNormalizer");

  _ajvErrorNormalizer = function () {
    return data;
  };

  return data;
}

function _dotenv() {
  const data = require("dotenv");

  _dotenv = function () {
    return data;
  };

  return data;
}

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

async function readConfig(configFile, request) {
  const data = await (0, _fsExtra().readFile)(configFile, "utf8");
  let result;

  if (configFile.endsWith(".json5") || configFile.endsWith(".json")) {
    result = require("json5").parse(data);
  } else if (configFile.endsWith(".js")) {
    result = require(configFile);

    if (result.default != null) {
      result = result.default;
    }

    if (typeof result === "function") {
      result = result(request);
    }

    result = await Promise.resolve(result);
  } else if (configFile.endsWith(".toml")) {
    result = require("toml").parse(data);
  } else {
    result = (0, _jsYaml().safeLoad)(data);
  }

  return {
    result,
    configFile
  };
}

async function findAndReadConfig(request) {
  const prefix = request.configFilename;

  for (const configFile of [`${prefix}.yml`, `${prefix}.yaml`, `${prefix}.json`, `${prefix}.json5`, `${prefix}.toml`, `${prefix}.js`]) {
    const data = await orNullIfFileNotExist(readConfig(path.join(request.projectDir, configFile), request));

    if (data != null) {
      return data;
    }
  }

  return null;
}

function orNullIfFileNotExist(promise) {
  return orIfFileNotExist(promise, null);
}

function orIfFileNotExist(promise, fallbackValue) {
  return promise.catch(e => {
    if (e.code === "ENOENT" || e.code === "ENOTDIR") {
      return fallbackValue;
    }

    throw e;
  });
}

async function loadConfig(request) {
  let packageMetadata = request.packageMetadata == null ? null : await request.packageMetadata.value;

  if (packageMetadata == null) {
    packageMetadata = await orNullIfFileNotExist((0, _fsExtra().readJson)(path.join(request.projectDir, "package.json")));
  }

  const data = packageMetadata == null ? null : packageMetadata[request.packageKey];
  return data == null ? findAndReadConfig(request) : {
    result: data,
    configFile: null
  };
}

function getConfig(request, configPath) {
  if (configPath == null) {
    return loadConfig(request);
  } else {
    return readConfig(path.resolve(request.projectDir, configPath), request);
  }
}

async function loadParentConfig(request, spec) {
  let isFileSpec;

  if (spec.startsWith("file:")) {
    spec = spec.substring("file:".length);
    isFileSpec = true;
  }

  let parentConfig = await orNullIfFileNotExist(readConfig(path.resolve(request.projectDir, spec), request));

  if (parentConfig == null && isFileSpec !== true) {
    let resolved = null;

    try {
      resolved = require.resolve(spec);
    } catch (e) {// ignore
    }

    if (resolved != null) {
      parentConfig = await readConfig(resolved, request);
    }
  }

  if (parentConfig == null) {
    throw new Error(`Cannot find parent config file: ${spec}`);
  }

  return parentConfig;
}

async function validateConfig(config, scheme, errorMessage) {
  const ajv = new (_ajv().default)({
    allErrors: true,
    coerceTypes: true,
    verbose: true,
    errorDataPath: "configuration"
  });

  require("ajv-keywords")(ajv, ["typeof"]);

  const schema = await scheme.value;
  const validator = ajv.compile(schema);

  if (!validator(config)) {
    const error = new Error(errorMessage((0, _ajvErrorNormalizer().normaliseErrorMessages)(validator.errors, schema), validator.errors));
    error.code = "ERR_CONFIG_INVALID";
    throw error;
  }
}

async function loadEnv(envFile) {
  const data = await orNullIfFileNotExist((0, _fsExtra().readFile)(envFile, "utf8"));

  if (data == null) {
    return null;
  }

  const parsed = (0, _dotenv().parse)(data);

  for (const key of Object.keys(parsed)) {
    if (!process.env.hasOwnProperty(key)) {
      process.env[key] = parsed[key];
    }
  }

  require("dotenv-expand")(parsed);

  return parsed;
} 
// __ts-babel@6.0.4
//# sourceMappingURL=main.js.map