"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
function genSequence(i) {
    let createIterable;
    if (typeof i === "function") {
        createIterable = i;
    }
    else {
        // this is typeof "object"
        createIterable = () => i;
    }
    function fnNext() {
        let iter;
        return () => {
            if (!iter) {
                iter = createIterable()[Symbol.iterator]();
            }
            const result = iter.next();
            if (result.done) {
                iter = undefined;
            }
            return result;
        };
    }
    const seq = {
        [Symbol.iterator]: () => createIterable()[Symbol.iterator](),
        next: fnNext(),
        //// Filters
        filter: (fnFilter) => genSequence(() => filter(fnFilter, createIterable())),
        skip: (n) => {
            return genSequence(() => skip(n, createIterable()));
        },
        take: (n) => {
            return genSequence(() => take(n, createIterable()));
        },
        //// Extenders
        concat: (j) => {
            return genSequence(() => concat(createIterable(), j));
        },
        concatMap: (fn) => {
            return genSequence(() => concatMap(fn, createIterable()));
        },
        //// Mappers
        combine: (fn, j) => {
            return genSequence(() => combine(fn, createIterable(), j));
        },
        map: (fn) => genSequence(() => map(fn, createIterable())),
        scan: (fnReduce, initValue) => {
            return genSequence(() => scan(createIterable(), fnReduce, initValue));
        },
        // Reducers
        all: (fnFilter) => {
            return all(fnFilter, createIterable());
        },
        any: (fnFilter) => {
            return any(fnFilter, createIterable());
        },
        count: () => {
            return count(createIterable());
        },
        first: (fnFilter, defaultValue) => {
            return first(fnFilter, defaultValue, createIterable());
        },
        forEach: (fn) => {
            return forEach(fn, createIterable());
        },
        max: (fnSelector) => {
            return max(fnSelector, createIterable());
        },
        min: (fnSelector) => {
            return min(fnSelector, createIterable());
        },
        reduce: (fnReduce, initValue) => {
            return reduce(fnReduce, initValue, createIterable());
        },
        reduceToSequence: (fnReduce, initialValue) => {
            return genSequence(reduce(fnReduce, initialValue, createIterable()));
        },
        //// Cast
        toArray: () => [...createIterable()],
        toIterable: () => {
            return toIterator(createIterable());
        },
    };
    return seq;
}
exports.genSequence = genSequence;
// Collection of entry points into GenSequence
exports.GenSequence = {
    genSequence,
    sequenceFromRegExpMatch,
    sequenceFromObject,
};
//// Filters
function* filter(fnFilter, i) {
    for (const v of i) {
        if (fnFilter(v)) {
            yield v;
        }
    }
}
exports.filter = filter;
function* skip(n, i) {
    let a = 0;
    for (const t of i) {
        if (a >= n) {
            yield t;
        }
        a += 1;
    }
}
exports.skip = skip;
function* take(n, i) {
    let a = 0;
    if (n) {
        for (const t of i) {
            if (a >= n) {
                break;
            }
            yield t;
            a += 1;
        }
    }
}
exports.take = take;
//// Extenders
/**
 * Concat two iterables together
 */
function* concat(i, j) {
    yield* i;
    yield* j;
}
exports.concat = concat;
function* concatMap(fn, i) {
    for (const t of i) {
        for (const u of fn(t)) {
            yield u;
        }
    }
}
exports.concatMap = concatMap;
//// Mappers
/**
 * Combine two iterables together using fnMap function.
 */
function* combine(fnMap, i, j) {
    const jit = j[Symbol.iterator]();
    for (const r of i) {
        const s = jit.next().value;
        yield fnMap(r, s);
    }
}
exports.combine = combine;
function map(fnMap, i) {
    function* fn(fnMap, i) {
        for (const v of i) {
            yield fnMap(v);
        }
    }
    if (i !== undefined) {
        return fn(fnMap, i);
    }
    return function (i) {
        return fn(fnMap, i);
    };
}
exports.map = map;
function* scan(i, fnReduce, initValue) {
    let index = 0;
    if (initValue === undefined) {
        // We need to create a new iterable to prevent for...of from restarting an array.
        index = 1;
        const iter = i[Symbol.iterator]();
        let r = iter.next();
        if (!r.done)
            yield r.value;
        initValue = r.value;
        i = makeIterable(iter);
    }
    let prevValue = initValue;
    for (const t of i) {
        const nextValue = fnReduce(prevValue, t, index);
        yield nextValue;
        prevValue = nextValue;
        index += 1;
    }
}
exports.scan = scan;
//// Reducers
function all(fn, i) {
    for (const t of i) {
        if (!fn(t)) {
            return false;
        }
    }
    return true;
}
exports.all = all;
function any(fn, i) {
    for (const t of i) {
        if (fn(t)) {
            return true;
        }
    }
    return false;
}
exports.any = any;
function count(i) {
    return reduce(p => p + 1, 0, i);
}
exports.count = count;
function first(fn, defaultValue, i) {
    fn = fn || (() => true);
    for (const t of i) {
        if (fn(t)) {
            return t;
        }
    }
    return defaultValue;
}
exports.first = first;
function forEach(fn, i) {
    let index = 0;
    for (const t of i) {
        fn(t, index);
        index += 1;
    }
}
exports.forEach = forEach;
function max(selector = (t => t), i) {
    return reduce((p, c) => selector(c) > selector(p) ? c : p, undefined, i);
}
exports.max = max;
function min(selector = (t => t), i) {
    return reduce((p, c) => selector(c) < selector(p) ? c : p, undefined, i);
}
exports.min = min;
function reduce(fnReduce, initialValue, i) {
    let index = 0;
    if (initialValue === undefined) {
        index = 1;
        const r = i[Symbol.iterator]().next();
        initialValue = r.value;
    }
    let prevValue = initialValue;
    for (const t of i) {
        const nextValue = fnReduce(prevValue, t, index);
        prevValue = nextValue;
        index += 1;
    }
    return prevValue;
}
exports.reduce = reduce;
//// Cast
function* toIterator(i) {
    yield* i;
}
exports.toIterator = toIterator;
//// Utilities
/**
 * Convert an Iterator into an IterableIterator
 */
function makeIterable(i) {
    function* iterate() {
        for (let r = i.next(); !r.done; r = i.next()) {
            yield r.value;
        }
    }
    return iterate();
}
exports.makeIterable = makeIterable;
function scanMap(accFn, init) {
    let acc = init;
    let first = true;
    return function (value) {
        if (first && acc === undefined) {
            first = false;
            acc = value;
            return acc;
        }
        acc = accFn(acc, value);
        return acc;
    };
}
exports.scanMap = scanMap;
function* objectIterator(t) {
    const keys = new Set(Object.keys(t));
    for (const k in t) {
        if (keys.has(k)) {
            yield [k, t[k]];
        }
    }
}
exports.objectIterator = objectIterator;
function objectToSequence(t) {
    return sequenceFromObject(t);
}
exports.objectToSequence = objectToSequence;
function sequenceFromObject(t) {
    return genSequence(() => objectIterator(t));
}
exports.sequenceFromObject = sequenceFromObject;
function sequenceFromRegExpMatch(pattern, text) {
    function* doMatch() {
        const regex = new RegExp(pattern);
        let match;
        let lastIndex = undefined;
        while (match = regex.exec(text)) {
            // Make sure it stops if the index does not move forward.
            if (match.index === lastIndex) {
                break;
            }
            lastIndex = match.index;
            yield match;
        }
    }
    return genSequence(() => doMatch());
}
exports.sequenceFromRegExpMatch = sequenceFromRegExpMatch;
exports.default = genSequence;
//# sourceMappingURL=GenSequence.js.map