"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const glob = require("glob");
const cspell = require("cspell-lib");
const fsp = require("fs-extra");
const path = require("path");
const commentJson = require("comment-json");
const util = require("./util/util");
const cspell_lib_1 = require("cspell-lib");
const Validator = require("cspell-lib");
const getStdin = require("get-stdin");
var cspell_lib_2 = require("cspell-lib");
exports.IncludeExcludeFlag = cspell_lib_2.IncludeExcludeFlag;
const cspell_glob_1 = require("cspell-glob");
// cspell:word nocase
const UTF8 = 'utf8';
const STDIN = 'stdin';
exports.MessageTypes = {
    Debug: 'Debug',
    Info: 'Info',
    Progress: 'Progress',
};
const defaultMinimatchOptions = { nocase: true };
const defaultConfigGlob = '{cspell.json,.cspell.json}';
const defaultConfigGlobOptions = defaultMinimatchOptions;
const nullEmitter = () => { };
class CSpellApplicationConfiguration {
    constructor(files, options, emitters) {
        this.files = files;
        this.options = options;
        this.emitters = emitters;
        this.configGlob = defaultConfigGlob;
        this.configGlobOptions = defaultConfigGlobOptions;
        this.root = path.resolve(options.root || process.cwd());
        this.info = emitters.info || nullEmitter;
        this.debug = emitters.debug || ((msg) => this.info(msg, exports.MessageTypes.Debug));
        this.configGlob = options.config || this.configGlob;
        this.configGlobOptions = options.config ? {} : this.configGlobOptions;
        this.excludes = calcExcludeGlobInfo(this.root, options.exclude);
        this.logIssue = emitters.issue || nullEmitter;
        this.local = options.local || '';
        this.uniqueFilter = options.unique
            ? util.uniqueFilterFnGenerator((issue) => issue.text)
            : () => true;
    }
}
exports.CSpellApplicationConfiguration = CSpellApplicationConfiguration;
function lint(files, options, emitters) {
    const cfg = new CSpellApplicationConfiguration(files, options, emitters);
    return runLint(cfg);
}
exports.lint = lint;
function runLint(cfg) {
    return run();
    async function processFile(fileInfo, configInfo) {
        const settingsFromCommandLine = util.clean({
            languageId: cfg.options.languageId || undefined,
            language: cfg.local || undefined,
        });
        const { filename, text } = fileInfo;
        const info = calcFinalConfigInfo(configInfo, settingsFromCommandLine, filename, text);
        const config = info.configInfo.config;
        const source = info.configInfo.source;
        cfg.debug(`Filename: ${filename}, Extension: ${path.extname(filename)}, LanguageIds: ${info.languageIds.toString()}`);
        if (!info.configInfo.config.enabled)
            return 0;
        const debugCfg = { config: Object.assign({}, config, { source: null }), source };
        cfg.debug(commentJson.stringify(debugCfg, undefined, 2));
        const startTime = Date.now();
        const wordOffsets = await cspell.validateText(text, info.configInfo.config);
        const issues = cspell.Text.calculateTextDocumentOffsets(filename, text, wordOffsets);
        const elapsed = (Date.now() - startTime) / 1000.0;
        const dictionaries = config.dictionaries || [];
        cfg.info(`Checking: ${filename}, File type: ${config.languageId}, Language: ${config.language} ... Issues: ${issues.length} ${elapsed}S`, exports.MessageTypes.Info);
        cfg.info(`Dictionaries Used: ${dictionaries.join(', ')}`, exports.MessageTypes.Info);
        issues
            .filter(cfg.uniqueFilter)
            .forEach((issue) => cfg.logIssue(issue));
        return issues.length;
    }
    /**
     * The file loader is written this way to cause files to be loaded in parallel while the previous one is being processed.
     * @param fileNames names of files to load one at a time.
     */
    function* fileLoader(fileNames) {
        for (const filename of fileNames) {
            // console.log(`${Date.now()} Start reading       ${filename}`);
            const file = readFileInfo(filename);
            // console.log(`${Date.now()} Waiting for request ${filename}`);
            yield file;
            // console.log(`${Date.now()} Yield               ${filename}`);
        }
    }
    async function processFiles(files, configInfo) {
        const status = {
            files: 0,
            filesWithIssues: new Set(),
            issues: 0,
        };
        for (const fileP of files) {
            const file = await fileP;
            if (!file || !file.text) {
                continue;
            }
            const r = await processFile(file, configInfo);
            status.files += 1;
            if (r) {
                status.filesWithIssues.add(file.filename);
                status.issues += r;
            }
        }
        return status;
    }
    async function run() {
        header();
        const configFiles = (await globP(cfg.configGlob, cfg.configGlobOptions)).filter(util.uniqueFn());
        cfg.info(`Config Files Found:\n    ${configFiles.join('\n    ')}\n`, exports.MessageTypes.Info);
        const config = cspell.readSettingsFiles(configFiles);
        const configInfo = { source: configFiles.join(' || '), config };
        // Get Exclusions from the config files.
        const { root } = cfg;
        const globOptions = { root, cwd: root };
        const exclusionGlobs = extractGlobExcludesFromConfig(root, configInfo.source, configInfo.config).concat(cfg.excludes);
        const files = filterFiles(await findFiles(cfg.files, globOptions), exclusionGlobs);
        return processFiles(fileLoader(files), configInfo);
    }
    function header() {
        cfg.info(`
cspell;
Date: ${(new Date()).toUTCString()}
Options:
    verbose:   ${yesNo(!!cfg.options.verbose)}
    config:    ${cfg.configGlob}
    exclude:   ${extractPatterns(cfg.excludes).map(a => a.glob).join('\n             ')}
    files:     ${cfg.files}
    wordsOnly: ${yesNo(!!cfg.options.wordsOnly)}
    unique:    ${yesNo(!!cfg.options.unique)}
`, exports.MessageTypes.Info);
    }
    function isExcluded(filename, globs) {
        const { root } = cfg;
        const absFilename = path.resolve(root, filename);
        for (const glob of globs) {
            const m = glob.matcher.matchEx(absFilename);
            if (m.matched) {
                cfg.info(`Excluded File: ${path.relative(root, absFilename)}; Excluded by ${m.glob} from ${glob.source}`, exports.MessageTypes.Info);
                return true;
            }
        }
        return false;
    }
    function filterFiles(files, excludeGlobs) {
        const excludeInfo = extractPatterns(excludeGlobs)
            .map(g => `Glob: ${g.glob} from ${g.source}`);
        cfg.info(`Exclusion Globs: \n    ${excludeInfo.join('\n    ')}\n`, exports.MessageTypes.Info);
        const result = files.filter(filename => !isExcluded(filename, excludeGlobs));
        return result;
    }
}
function extractPatterns(globs) {
    const r = globs
        .reduce((info, g) => {
        const source = g.source;
        const patterns = typeof g.matcher.patterns === 'string' ? [g.matcher.patterns] : g.matcher.patterns;
        return info.concat(patterns.map(glob => ({ glob, source })));
    }, []);
    return r;
}
async function trace(words, options) {
    const configGlob = options.config || defaultConfigGlob;
    const configGlobOptions = options.config ? {} : defaultConfigGlobOptions;
    const configFiles = (await globP(configGlob, configGlobOptions)).filter(util.uniqueFn());
    const config = cspell.mergeSettings(cspell.getDefaultSettings(), cspell.getGlobalSettings(), cspell.readSettingsFiles(configFiles));
    const results = await cspell_lib_1.traceWords(words, config);
    return results;
}
exports.trace = trace;
async function checkText(filename, options) {
    const configGlob = options.config || defaultConfigGlob;
    const configGlobOptions = options.config ? {} : defaultConfigGlobOptions;
    const pSettings = globP(configGlob, configGlobOptions).then(filenames => ({ source: filenames[0], config: cspell.readSettingsFiles(filenames) }));
    const [foundSettings, text] = await Promise.all([pSettings, readFile(filename)]);
    const settingsFromCommandLine = util.clean({
        languageId: options.languageId || undefined,
        local: options.local || undefined,
    });
    const info = calcFinalConfigInfo(foundSettings, settingsFromCommandLine, filename, text);
    return Validator.checkText(text, info.configInfo.config);
}
exports.checkText = checkText;
function createInit(_) {
    return Promise.resolve();
}
exports.createInit = createInit;
const defaultExcludeGlobs = [
    'node_modules/**'
];
function readFileInfo(filename, encoding = UTF8) {
    const pText = filename === STDIN ? getStdin() : fsp.readFile(filename, encoding);
    return pText.then(text => ({ text, filename }), error => {
        return error.code === 'EISDIR'
            ? Promise.resolve({ text: '', filename })
            : Promise.reject(Object.assign({}, error, { message: `Error reading file: "${filename}"` }));
    });
}
function readFile(filename, encoding = UTF8) {
    return readFileInfo(filename, encoding).then(info => info.text);
}
/**
 * Looks for matching glob patterns or stdin
 * @param globPatterns patterns or stdin
 */
async function findFiles(globPatterns, options) {
    const globPats = globPatterns.filter(filename => filename !== STDIN);
    const stdin = globPats.length < globPatterns.length ? [STDIN] : [];
    const globs = globPats.length ? (await globP(globPats, options)) : [];
    const cwd = options.cwd || process.cwd();
    return stdin.concat(globs.map(filename => path.resolve(cwd, filename)));
}
function calcExcludeGlobInfo(root, commandLineExclude) {
    const commandLineExcludes = {
        globs: commandLineExclude ? commandLineExclude.split(/\s+/g) : [],
        source: 'arguments'
    };
    const defaultExcludes = {
        globs: defaultExcludeGlobs,
        source: 'default'
    };
    const choice = commandLineExcludes.globs.length ? commandLineExcludes : defaultExcludes;
    const matcher = new cspell_glob_1.GlobMatcher(choice.globs, root);
    return [{
            matcher,
            source: choice.source
        }];
}
function extractGlobExcludesFromConfig(root, source, config) {
    if (!config.ignorePaths || !config.ignorePaths.length) {
        return [];
    }
    const matcher = new cspell_glob_1.GlobMatcher(config.ignorePaths, root);
    return [{ source, matcher }];
}
function calcFinalConfigInfo(configInfo, settingsFromCommandLine, filename, text) {
    const ext = path.extname(filename);
    const fileSettings = cspell.calcOverrideSettings(configInfo.config, path.resolve(filename));
    const settings = cspell.mergeSettings(cspell.getDefaultSettings(), cspell.getGlobalSettings(), fileSettings, settingsFromCommandLine);
    const languageIds = settings.languageId ? [settings.languageId] : cspell.getLanguagesForExt(ext);
    const config = cspell.constructSettingsForText(settings, text, languageIds);
    return { configInfo: Object.assign({}, configInfo, { config }), filename, text, languageIds };
}
function yesNo(value) {
    return value ? 'Yes' : 'No';
}
function globP(pattern, options) {
    const globPattern = typeof pattern === 'string'
        ? pattern
        : pattern.length > 1
            ? `{${pattern.join(',')}}`
            : (pattern[0] || '');
    if (!globPattern) {
        return Promise.resolve([]);
    }
    return new Promise((resolve, reject) => {
        const cb = (err, matches) => {
            if (err) {
                reject(err);
            }
            resolve(matches);
        };
        if (options) {
            glob(globPattern, options, cb);
        }
        else {
            glob(globPattern, cb);
        }
    });
}
//# sourceMappingURL=application.js.map