"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const wordListHelper_1 = require("../wordListHelper");
const SpellingDictionary_1 = require("./SpellingDictionary");
const path = require("path");
const gensequence_1 = require("gensequence");
const fileReader_1 = require("../util/fileReader");
const loaders = {
    S: loadSimpleWordList,
    W: loadWordList,
    C: loadCodeWordList,
    T: loadTrie,
    default: loadSimpleWordList,
};
const dictionaryCache = new Map();
function loadDictionary(uri, options) {
    const loaderType = determineType(uri, options);
    const key = [uri, loaderType].join('|');
    if (!dictionaryCache.has(key)) {
        dictionaryCache.set(key, load(uri, options));
    }
    return dictionaryCache.get(key);
}
exports.loadDictionary = loadDictionary;
function determineType(uri, options) {
    const defType = uri.endsWith('.trie.gz') ? 'T' : uri.endsWith('.txt.gz') ? 'S' : 'C';
    const { type = defType } = options;
    const regTrieTest = /\.trie\b/i;
    return regTrieTest.test(uri) ? 'T' : type;
}
function load(uri, options) {
    const type = determineType(uri, options);
    const loader = loaders[type] || loaders.default;
    return loader(uri, options);
}
async function loadSimpleWordList(filename, options) {
    try {
        const lines = await fileReader_1.readLines(filename);
        return SpellingDictionary_1.createSpellingDictionary(lines, path.basename(filename), filename, options);
    }
    catch (e) {
        return Promise.reject(e);
    }
}
async function loadWordList(filename, options) {
    const lines = gensequence_1.genSequence(await fileReader_1.readLines(filename));
    const words = lines.concatMap(wordListHelper_1.splitLineIntoWords);
    return SpellingDictionary_1.createSpellingDictionary(words, path.basename(filename), filename, options);
}
async function loadCodeWordList(filename, options) {
    const lines = gensequence_1.genSequence(await fileReader_1.readLines(filename));
    const words = lines.concatMap(wordListHelper_1.splitLineIntoCodeWords);
    return SpellingDictionary_1.createSpellingDictionary(words, path.basename(filename), filename, options);
}
async function loadTrie(filename, options) {
    return SpellingDictionary_1.createSpellingDictionaryTrie(await wordListHelper_1.loadWordsNoError(filename), path.basename(filename), filename, options);
}
//# sourceMappingURL=DictionaryLoader.js.map