"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const mm = require("micromatch");
class GlobMatcher {
    /**
     * Construct a `.gitignore` emulator
     * @param patterns the contents of a `.gitignore` style file or an array of individual glob rules.
     * @param root the working directory
     */
    constructor(patterns, root) {
        this.patterns = patterns;
        this.root = root;
        this.matchEx = buildMatcherFn(patterns, root);
    }
    /**
     * Check to see if a filename matches any of the globs.
     * @param filename full path of the file to check.
     */
    match(filename) {
        return this.matchEx(filename).matched;
    }
}
exports.GlobMatcher = GlobMatcher;
/**
 * This function attempts to emulate .gitignore functionality as much as possible.
 * @param patterns the contents of a .gitignore style file or an array of individual glob rules.
 * @param root the working directory
 * @returns a function given a filename returns if it matches.
 */
function buildMatcherFn(patterns, root) {
    if (typeof patterns == 'string') {
        patterns = patterns.split(/\r?\n/g);
    }
    const r = (root || '').replace(/\/$/, '');
    const rules = patterns
        .map(p => p.trim())
        .map((p, index) => ({ glob: p, index }))
        .filter(r => !!r.glob)
        .map(({ glob, index }) => {
        const matchNeg = glob.match(/^!+/);
        const isNeg = matchNeg && (matchNeg[0].length & 1) && true || false;
        const pattern = mutations.reduce((p, [regex, replace]) => p.replace(regex, replace), glob);
        const reg = mm.makeRe(pattern);
        const fn = (filename) => {
            const match = filename.match(reg);
            return !!match;
        };
        return { glob, index, isNeg, fn, reg };
    });
    const negRules = rules.filter(r => r.isNeg);
    const posRules = rules.filter(r => !r.isNeg);
    const fn = (filename) => {
        filename = filename.replace(/^[^/]/, '/$&');
        const offset = r === filename.slice(0, r.length) ? r.length : 0;
        const fname = filename.slice(offset);
        for (const rule of negRules) {
            if (rule.fn(fname)) {
                return { matched: false, glob: rule.glob, index: rule.index, isNeg: rule.isNeg };
            }
        }
        for (const rule of posRules) {
            if (rule.fn(fname)) {
                return { matched: true, glob: rule.glob, index: rule.index, isNeg: rule.isNeg };
            }
        }
        return { matched: false };
    };
    return fn;
}
const mutations = [
    [/^!+/, ''],
    [/^[^/#][^/]*$/, '**/{$&,$&/**}',],
    [/\/$/, '$&**',],
    [/^(([^/*])|([*][^*])).*[/]/, '/$&',],
];
//# sourceMappingURL=GlobMatcher.js.map