"use strict";
exports.__esModule = true;
var Base64Message_1 = require("./Model/Base64Message");
var DataSource_1 = require("./DataSource");
var events_1 = require("../../events");
var ConnectionManager = /** @class */ (function () {
    function ConnectionManager() {
        var _this = this;
        this.connections = {};
        this.handleConnectionRequest = function (event) {
            var connectionId = event.id;
            // Prevent double connections when reloading
            if (_this.connections[connectionId]) {
                _this.removeConnection(connectionId);
            }
            var options = event.options;
            var connection = new DataSource_1.MqttSource();
            _this.connections[connectionId] = connection;
            var connectionStateEvent = events_1.makeConnectionStateEvent(connectionId);
            connection.stateMachine.onUpdate.subscribe(function (state) {
                events_1.backendEvents.emit(connectionStateEvent, state);
            });
            connection.connect(options);
            _this.handleNewMessagesForConnection(connectionId, connection);
            events_1.backendEvents.subscribe(events_1.makePublishEvent(connectionId), function (msg) {
                _this.connections[connectionId].publish(msg);
            });
        };
    }
    ConnectionManager.prototype.handleNewMessagesForConnection = function (connectionId, connection) {
        var messageEvent = events_1.makeConnectionMessageEvent(connectionId);
        connection.onMessage(function (topic, payload, packet) {
            var buffer = payload;
            if (buffer.length > 20000) {
                buffer = buffer.slice(0, 20000);
            }
            events_1.backendEvents.emit(messageEvent, {
                topic: topic,
                payload: Base64Message_1.Base64Message.fromBuffer(buffer),
                qos: packet.qos,
                retain: packet.retain
            });
        });
    };
    ConnectionManager.prototype.manageConnections = function () {
        var _this = this;
        events_1.backendEvents.subscribe(events_1.addMqttConnectionEvent, this.handleConnectionRequest);
        events_1.backendEvents.subscribe(events_1.removeConnection, function (connectionId) {
            _this.removeConnection(connectionId);
        });
    };
    ConnectionManager.prototype.removeConnection = function (connectionId) {
        var connection = this.connections[connectionId];
        if (connection) {
            events_1.backendEvents.unsubscribeAll(events_1.makePublishEvent(connectionId));
            connection.disconnect();
            delete this.connections[connectionId];
            connection.stateMachine.onUpdate.removeAllListeners();
        }
    };
    ConnectionManager.prototype.closeAllConnections = function () {
        var _this = this;
        Object.keys(this.connections).forEach(function (connectionId) { return _this.removeConnection(connectionId); });
    };
    return ConnectionManager;
}());
exports.ConnectionManager = ConnectionManager;
//# sourceMappingURL=index.js.map