"use strict";
exports.__esModule = true;
var Url = require("url");
var mqtt_1 = require("mqtt");
var _1 = require("./");
var Base64Message_1 = require("../Model/Base64Message");
var MqttSource = /** @class */ (function () {
    function MqttSource() {
        this.stateMachine = new _1.DataSourceStateMachine();
        this.topicSeparator = '/';
    }
    MqttSource.prototype.onMessage = function (messageCallback) {
        this.messageCallback = messageCallback;
    };
    MqttSource.prototype.connect = function (options) {
        var _this = this;
        this.stateMachine.setConnecting();
        var urlStr = options.tls ? options.url.replace(/^(mqtt|ws):/, '$1s:') : options.url;
        var url;
        try {
            url = Url.parse(urlStr);
        }
        catch (error) {
            this.stateMachine.setError(error);
            throw error;
        }
        var client = mqtt_1.connect(url, {
            resubscribe: false,
            rejectUnauthorized: options.certValidation,
            username: options.username,
            password: options.password,
            clientId: options.clientId,
            servername: options.tls ? url.hostname : undefined,
            ca: options.certificateAuthority ? Buffer.from(options.certificateAuthority, 'base64') : undefined,
            cert: options.clientCertificate ? Buffer.from(options.clientCertificate, 'base64') : undefined,
            key: options.clientKey ? Buffer.from(options.clientKey, 'base64') : undefined
        });
        this.client = client;
        client.on('error', function (error) {
            console.log(error);
            _this.stateMachine.setError(error);
        });
        client.on('close', function () {
            _this.stateMachine.setConnected(false);
        });
        client.on('end', function () {
            _this.stateMachine.setConnected(false);
        });
        client.on('reconnect', function () {
            _this.stateMachine.setConnecting();
        });
        client.on('connect', function () {
            _this.stateMachine.setConnected(true);
            options.subscriptions.forEach(function (subscription) {
                client.subscribe(subscription, function (err) {
                    if (err) {
                        _this.stateMachine.setError(err);
                    }
                });
            });
        });
        client.on('message', function (topic, message, packet) {
            _this.messageCallback && _this.messageCallback(topic, message, packet);
        });
        return this.stateMachine;
    };
    MqttSource.prototype.publish = function (msg) {
        if (this.client) {
            this.client.publish(msg.topic, msg.payload ? Base64Message_1.Base64Message.toUnicodeString(msg.payload) : '', {
                qos: msg.qos,
                retain: msg.retain
            });
        }
    };
    MqttSource.prototype.disconnect = function () {
        this.client && this.client.end();
    };
    return MqttSource;
}());
exports.MqttSource = MqttSource;
//# sourceMappingURL=MqttSource.js.map