import _extends from "@babel/runtime/helpers/extends";
import _slicedToArray from "@babel/runtime/helpers/slicedToArray";
import _objectWithoutProperties from "@babel/runtime/helpers/objectWithoutProperties";
import React from 'react';
import PropTypes from 'prop-types';
import debounce from 'debounce'; // < 1kb payload overhead when lodash/debounce is > 3kb.

import { useForkRef } from '../utils/reactHelpers';

function getStyleValue(computedStyle, property) {
  return parseInt(computedStyle[property], 10) || 0;
}

var useEnhancedEffect = typeof window !== 'undefined' ? React.useLayoutEffect : React.useEffect;
var styles = {
  /* Styles applied to the shadow textarea element. */
  shadow: {
    // Visibility needed to hide the extra text area on iPads
    visibility: 'hidden',
    // Remove from the content flow
    position: 'absolute',
    // Ignore the scrollbar width
    overflow: 'hidden',
    height: '0'
  }
};
/**
 * @ignore - internal component.
 *
 * To make public in v4+.
 */

var Textarea = React.forwardRef(function Textarea(props, ref) {
  var onChange = props.onChange,
      rows = props.rows,
      rowsMax = props.rowsMax,
      style = props.style,
      value = props.value,
      other = _objectWithoutProperties(props, ["onChange", "rows", "rowsMax", "style", "value"]);

  var _React$useRef = React.useRef(value != null),
      isControlled = _React$useRef.current;

  var inputRef = React.useRef(null);

  var _React$useState = React.useState({}),
      _React$useState2 = _slicedToArray(_React$useState, 2),
      state = _React$useState2[0],
      setState = _React$useState2[1];

  var shadowRef = React.useRef(null);
  var handleRef = useForkRef(ref, inputRef);
  var syncHeight = React.useCallback(function () {
    var input = inputRef.current;
    var inputShallow = shadowRef.current;
    var computedStyle = window.getComputedStyle(input);
    inputShallow.style.width = computedStyle.width;
    inputShallow.value = input.value || props.placeholder || 'x'; // The height of the inner content

    var innerHeight = inputShallow.scrollHeight;
    var boxSizing = computedStyle['box-sizing']; // Measure height of a textarea with a single row

    inputShallow.value = 'x';
    var singleRowHeight = inputShallow.scrollHeight; // The height of the outer content

    var outerHeight = innerHeight;

    if (rows != null) {
      outerHeight = Math.max(Number(rows) * singleRowHeight, outerHeight);
    }

    if (rowsMax != null) {
      outerHeight = Math.min(Number(rowsMax) * singleRowHeight, outerHeight);
    }

    outerHeight = Math.max(outerHeight, singleRowHeight);

    if (boxSizing === 'content-box') {
      outerHeight -= getStyleValue(computedStyle, 'padding-bottom') + getStyleValue(computedStyle, 'padding-top');
    } else if (boxSizing === 'border-box') {
      outerHeight += getStyleValue(computedStyle, 'border-bottom-width') + getStyleValue(computedStyle, 'border-top-width');
    }

    setState(function (prevState) {
      // Need a large enough different to update the height.
      // This prevents infinite rendering loop.
      if (innerHeight > 0 && Math.abs((prevState.innerHeight || 0) - innerHeight) > 1) {
        return {
          innerHeight: innerHeight,
          outerHeight: outerHeight
        };
      }

      return prevState;
    });
  }, [setState, rows, rowsMax, props.placeholder]);
  React.useEffect(function () {
    var handleResize = debounce(function () {
      syncHeight();
    }, 166); // Corresponds to 10 frames at 60 Hz.

    window.addEventListener('resize', handleResize);
    return function () {
      handleResize.clear();
      window.removeEventListener('resize', handleResize);
    };
  }, [syncHeight]);
  useEnhancedEffect(function () {
    syncHeight();
  });

  var handleChange = function handleChange(event) {
    if (!isControlled) {
      syncHeight();
    }

    if (onChange) {
      onChange(event);
    }
  };

  return React.createElement(React.Fragment, null, React.createElement("textarea", _extends({
    value: value,
    onChange: handleChange,
    ref: handleRef,
    style: _extends({
      height: state.outerHeight,
      overflow: state.outerHeight === state.innerHeight ? 'hidden' : null
    }, style)
  }, other)), React.createElement("textarea", {
    "aria-hidden": true,
    className: props.className,
    readOnly: true,
    ref: shadowRef,
    tabIndex: -1,
    style: _extends({}, styles.shadow, style)
  }));
});
process.env.NODE_ENV !== "production" ? Textarea.propTypes = {
  /**
   * @ignore
   */
  className: PropTypes.string,

  /**
   * @ignore
   */
  onChange: PropTypes.func,

  /**
   * @ignore
   */
  placeholder: PropTypes.string,

  /**
   * Minimum umber of rows to display.
   */
  rows: PropTypes.oneOfType([PropTypes.string, PropTypes.number]),

  /**
   * Maximum number of rows to display.
   */
  rowsMax: PropTypes.oneOfType([PropTypes.string, PropTypes.number]),

  /**
   * @ignore
   */
  style: PropTypes.object,

  /**
   * @ignore
   */
  value: PropTypes.any
} : void 0;
export default Textarea;