import React from 'react';
import ReactDOM from 'react-dom';
import PropTypes from 'prop-types';
import { useForkRef } from '../utils/reactHelpers';
import { exactProp } from '@material-ui/utils';

function getContainer(container) {
  container = typeof container === 'function' ? container() : container; // #StrictMode ready

  return ReactDOM.findDOMNode(container);
}

const useEnhancedEffect = typeof window !== 'undefined' ? React.useLayoutEffect : React.useEffect;
/**
 * Portals provide a first-class way to render children into a DOM node
 * that exists outside the DOM hierarchy of the parent component.
 */

const Portal = React.forwardRef(function Portal(props, ref) {
  const {
    children,
    container,
    disablePortal,
    onRendered
  } = props;
  const [mountNode, setMountNode] = React.useState(null);
  const childRef = React.useRef(null);
  const handleRef = useForkRef(children.ref, childRef);
  useEnhancedEffect(() => {
    if (!disablePortal) {
      setMountNode(getContainer(container) || document.body);
    }
  }, [container, disablePortal]);
  useEnhancedEffect(() => {
    if (onRendered && mountNode) {
      onRendered();
    }
  }, [mountNode, onRendered]);
  React.useImperativeHandle(ref, () => mountNode || childRef.current, [mountNode]);

  if (disablePortal) {
    React.Children.only(children);
    return React.cloneElement(children, {
      ref: handleRef
    });
  }

  return mountNode ? ReactDOM.createPortal(children, mountNode) : mountNode;
});
process.env.NODE_ENV !== "production" ? Portal.propTypes = {
  /**
   * The children to render into the `container`.
   */
  children: PropTypes.node.isRequired,

  /**
   * A node, component instance, or function that returns either.
   * The `container` will have the portal children appended to it.
   * By default, it uses the body of the top-level document object,
   * so it's simply `document.body` most of the time.
   */
  container: PropTypes.oneOfType([PropTypes.object, PropTypes.func]),

  /**
   * Disable the portal behavior.
   * The children stay within it's parent DOM hierarchy.
   */
  disablePortal: PropTypes.bool,

  /**
   * Callback fired once the children has been mounted into the `container`.
   */
  onRendered: PropTypes.func
} : void 0;
Portal.defaultProps = {
  disablePortal: false
};

if (process.env.NODE_ENV !== 'production') {
  // eslint-disable-next-line
  Portal['propTypes' + ''] = exactProp(Portal.propTypes);
}

export default Portal;